//
//  See LICENSE folder for this template’s licensing information.
//
//  Abstract:
//   playground ライブビューとコード間のメッセージハンドリング
//   playground BluetoothによるconnectionViewの表示
import UIKit
import SpriteKit
import PlaygroundSupport
import AVFoundation
import WebKit
protocol PlotSceneDelegate: AnyObject {
    func animationDidFinish()
}

public class PlotViewController: UIViewController, PlotSceneDelegate, WKNavigationDelegate {
    
    var webView: WKWebView!
    var isWebViewVisible = false // 初期状態ではWebViewを非表示
    
    private var skView: SKView!
    private var plotScene: PlotScene!  // SKSceneを管理する変数
    private var webViewButton: UIButton! //WebViewボタン
    private var rulerButton: UIButton! //定規ボタン
    private var protractButton: UIButton! //分度器ボタン
    private let minScale: CGFloat = 0.5  // 最小縮小率
    private let maxScale: CGFloat = 10.0  // 最大拡大率
    private let p_minScale: CGFloat = 0.3  // 分度器最小縮小率
    private let p_maxScale: CGFloat = 1.0  // 分度器最大拡大率
    private var currentScale: CGFloat = 1.0  // 現在のスケール（初期は1.0）
    private var p_currentScale: CGFloat = 0.5  // 分度器の現在のスケール（初期は1.0）
    private var lastScale: CGFloat = 0.5
    private var p_lastScale: CGFloat = 0.5 //分度器用
    private var protractResetButton: UIButton!
    private var rulerResetButton: UIButton!
    private var angleLabel: UILabel!
    private var rulerAngleLabel: UILabel!
    private var lastTranslation = CGPoint.zero //パンのための現在の位置
    
    // 初期化
    override public func viewDidLoad() {
        super.viewDidLoad()
        
        // WKWebViewを作成
        let webConfiguration = WKWebViewConfiguration()
        webView = WKWebView(frame: CGRect.zero, configuration: webConfiguration)
        webView.navigationDelegate = self
        webView.translatesAutoresizingMaskIntoConstraints = false
        webView.isHidden = true
        view.addSubview(webView)

        NSLayoutConstraint.activate([
            webView.topAnchor.constraint(equalTo: view.topAnchor, constant: 130),
            webView.bottomAnchor.constraint(equalTo: view.bottomAnchor, constant: -90),
            webView.leadingAnchor.constraint(equalTo: view.safeAreaLayoutGuide.leadingAnchor),
            webView.trailingAnchor.constraint(equalTo: view.safeAreaLayoutGuide.trailingAnchor)
        ])
        
        // ローカルファイルのURLを設定
        if let fileURL = Bundle.main.url(forResource: "index", withExtension: "html") {
            webView.loadFileURL(fileURL, allowingReadAccessTo: fileURL.deletingLastPathComponent())
        }
        
        //PlaygroundPage.current.needsIndefiniteExecution = true //実行しっぱなしにする。(falseだと、描画途中でも実行が終了するため）
        skView = SKView()
        skView.isUserInteractionEnabled = true
        skView.clipsToBounds = true  // はみ出さないようにする
        skView.backgroundColor = .black
        skView.translatesAutoresizingMaskIntoConstraints = false
        self.view.addSubview(skView)

        // SKViewのサイズが画面の小さい方に合わせる（正方形）
        let sideLength = min(self.view.bounds.width, self.view.bounds.height)
        skView.frame = CGRect(x: 0, y: 0, width: sideLength, height: sideLength)

        plotScene = PlotScene(size: CGSize(width: sideLength, height: sideLength))  // 正方形に設定
        plotScene.scaleMode = .aspectFill
        plotScene.plotDelegate = self
        skView.presentScene(plotScene)

        // ピンチジェスチャー
        let pinchGesture = UIPinchGestureRecognizer(target: self, action: #selector(handlePinch(_:)))
        skView.addGestureRecognizer(pinchGesture)
        
        // パンジェスチャー
        let panGesture = UIPanGestureRecognizer(target: self, action: #selector(handlePan(_:)))
        skView.addGestureRecognizer(panGesture)
        
        //ローテーションゼスチャー
        // 二本指回転ジェスチャーの設定
        let rotationRecognizer = UIRotationGestureRecognizer(target: self, action: #selector(handleRotation(_:)))
        skView.addGestureRecognizer(rotationRecognizer)

        
        // タップで背景色切り替え
        let tapGesture = UITapGestureRecognizer(target: self, action: #selector(togglePenVisible))
        skView.addGestureRecognizer(tapGesture)
        
        // 背景切り替えボタンを作成
        webViewButton = UIButton(type: .system)
        webViewButton.setTitle("ブロック", for: .normal)
        webViewButton.setTitleColor(.white, for: .normal)
        webViewButton.backgroundColor = .darkGray
        webViewButton.layer.cornerRadius = 8
        webViewButton.translatesAutoresizingMaskIntoConstraints = false
        self.view.addSubview(webViewButton)
        
        // 背景切り替えボタンを作成
        let toggleButton = UIButton(type: .system)
        toggleButton.setTitle("背景切替", for: .normal)
        toggleButton.setTitleColor(.white, for: .normal)
        toggleButton.backgroundColor = .darkGray
        toggleButton.layer.cornerRadius = 8
        toggleButton.translatesAutoresizingMaskIntoConstraints = false
        self.view.addSubview(toggleButton)
        
        // 分度器ON/OFFボタンを作成
        protractButton = UIButton(type: .system)
        protractButton.setTitle("分度器", for: .normal)
        protractButton.setTitleColor(.white, for: .normal)
        protractButton.backgroundColor = .darkGray
        protractButton.layer.cornerRadius = 8
        protractButton.translatesAutoresizingMaskIntoConstraints = false
        self.view.addSubview(protractButton)

        // 定規ON/OFFボタンを作成
        rulerButton = UIButton(type: .system)
        rulerButton.setTitle("定規", for: .normal)
        rulerButton.setTitleColor(.white, for: .normal)
        rulerButton.backgroundColor = .darkGray
        rulerButton.layer.cornerRadius = 8
        rulerButton.translatesAutoresizingMaskIntoConstraints = false
        self.view.addSubview(rulerButton)
        
        // 分度器リセットボタンを作成
        protractResetButton = UIButton(type: .system)
        let config = UIImage.SymbolConfiguration(pointSize: 20, weight: .regular)
        let image = UIImage(systemName: "arrow.counterclockwise.circle.fill", withConfiguration: config)
        protractResetButton.setImage(image, for: .normal)
        protractResetButton.tintColor = .lightGray  // アイコンの色をグレーに設定
        protractResetButton.isHidden = true    // 初期状態では非表示
        protractResetButton.clipsToBounds = true // 必要なければ削除しても可
        skView.addSubview(protractResetButton)
        protractResetButton.translatesAutoresizingMaskIntoConstraints = false
        NSLayoutConstraint.activate([
            protractResetButton.trailingAnchor.constraint(equalTo: skView.trailingAnchor, constant: -10),
            protractResetButton.topAnchor.constraint(equalTo: skView.topAnchor, constant: 20),
            protractResetButton.widthAnchor.constraint(equalToConstant: 40),
            protractResetButton.heightAnchor.constraint(equalToConstant: 40)
        ])
        // 分度器回転角度表示用ラベルを作成
        angleLabel = UILabel()
        angleLabel.font = UIFont.systemFont(ofSize: 16, weight: .medium)
        angleLabel.textColor = .lightGray
        angleLabel.text = "0°"  // 初期値
        angleLabel.textAlignment = .center
        angleLabel.isHidden = true
        angleLabel.translatesAutoresizingMaskIntoConstraints = false
        skView.addSubview(angleLabel)
        // レイアウト制約
        NSLayoutConstraint.activate([
            angleLabel.trailingAnchor.constraint(equalTo: protractResetButton.leadingAnchor, constant: -10),
            angleLabel.centerYAnchor.constraint(equalTo: protractResetButton.centerYAnchor),
        ])
        
        // 定規リセットボタンを作成
        rulerResetButton = UIButton(type: .system)
        let rulerConfig = UIImage.SymbolConfiguration(pointSize: 20, weight: .regular)
        let rulerImage = UIImage(systemName: "arrow.counterclockwise.circle.fill", withConfiguration: rulerConfig)
        rulerResetButton.setImage(rulerImage, for: .normal)
        rulerResetButton.tintColor = .lightGray
        rulerResetButton.isHidden = true
        rulerResetButton.clipsToBounds = true
        skView.addSubview(rulerResetButton)
        rulerResetButton.translatesAutoresizingMaskIntoConstraints = false

        NSLayoutConstraint.activate([
            rulerResetButton.leadingAnchor.constraint(equalTo: skView.leadingAnchor, constant: 10),
            rulerResetButton.topAnchor.constraint(equalTo: skView.topAnchor, constant: 20),
            rulerResetButton.widthAnchor.constraint(equalToConstant: 40),
            rulerResetButton.heightAnchor.constraint(equalToConstant: 40)
        ])

        // 定規角度表示用ラベルを作成
        rulerAngleLabel = UILabel()
        rulerAngleLabel.font = UIFont.systemFont(ofSize: 16, weight: .medium)
        rulerAngleLabel.textColor = .lightGray
        rulerAngleLabel.text = "0°"
        rulerAngleLabel.textAlignment = .center
        rulerAngleLabel.isHidden = true
        rulerAngleLabel.translatesAutoresizingMaskIntoConstraints = false
        skView.addSubview(rulerAngleLabel)

        NSLayoutConstraint.activate([
            rulerAngleLabel.leadingAnchor.constraint(equalTo: rulerResetButton.trailingAnchor, constant: 10),
            rulerAngleLabel.centerYAnchor.constraint(equalTo: rulerResetButton.centerYAnchor)
        ])
        
        //ヘルプボタンを作成
        let helpButton = UIButton(type: .system)
        helpButton.setTitle("使い方", for: .normal)
        helpButton.setTitleColor(.white, for: .normal)
        helpButton.backgroundColor = .darkGray
        helpButton.layer.cornerRadius = 8
        helpButton.translatesAutoresizingMaskIntoConstraints = false
        self.view.addSubview(helpButton)
        
        // レイアウト
        NSLayoutConstraint.activate([
            // helpボタンの制約
            helpButton.leadingAnchor.constraint(equalTo: view.safeAreaLayoutGuide.leadingAnchor, constant: 10),
            helpButton.topAnchor.constraint(equalTo: view.topAnchor, constant: 80),
            helpButton.widthAnchor.constraint(equalToConstant: 70),
            helpButton.heightAnchor.constraint(equalToConstant: 40),
            // helpボタンの制約
            webViewButton.leadingAnchor.constraint(equalTo: helpButton.trailingAnchor, constant: 10),
            webViewButton.topAnchor.constraint(equalTo: view.topAnchor, constant: 80),
            webViewButton.widthAnchor.constraint(equalToConstant: 70),
            webViewButton.heightAnchor.constraint(equalToConstant: 40),
            // toggleButtonの制約
            toggleButton.trailingAnchor.constraint(equalTo: view.safeAreaLayoutGuide.trailingAnchor, constant: -10),
            toggleButton.topAnchor.constraint(equalTo: view.topAnchor, constant: 80),
            toggleButton.widthAnchor.constraint(equalToConstant: 70),
            toggleButton.heightAnchor.constraint(equalToConstant: 40),
            // protractButtonの制約
            protractButton.trailingAnchor.constraint(equalTo: toggleButton.leadingAnchor, constant: -10),
            protractButton.topAnchor.constraint(equalTo: view.topAnchor, constant: 80),
            protractButton.widthAnchor.constraint(equalToConstant: 70),
            protractButton.heightAnchor.constraint(equalToConstant: 40),
            // rulerButtonの制約
            rulerButton.trailingAnchor.constraint(equalTo: protractButton.leadingAnchor, constant: -10),
            rulerButton.topAnchor.constraint(equalTo: view.topAnchor, constant: 80),
            rulerButton.widthAnchor.constraint(equalToConstant: 70),
            rulerButton.heightAnchor.constraint(equalToConstant: 40),
            // skViewの制約
            skView.leadingAnchor.constraint(equalTo: view.safeAreaLayoutGuide.leadingAnchor, constant: 0),
            skView.trailingAnchor.constraint(equalTo: view.safeAreaLayoutGuide.trailingAnchor, constant: 0),
            skView.topAnchor.constraint(equalTo: toggleButton.bottomAnchor, constant: 10),
            skView.bottomAnchor.constraint(equalTo: view.bottomAnchor, constant: -90)
        ])

        // ボタンのアクション
        webViewButton.addTarget(self, action: #selector(toggleWebView), for: .touchUpInside)
        helpButton.addTarget(self, action: #selector(showHelp), for: .touchUpInside)
        toggleButton.addTarget(self, action: #selector(toggleBackgroundButtonTapped), for: .touchUpInside)
        protractButton.addTarget(self, action: #selector(protractButtonTapped), for: .touchUpInside)
        rulerButton.addTarget(self, action: #selector(rulerButtonTapped), for: .touchUpInside)
        protractResetButton.addTarget(self, action: #selector(resetProtractorTapped), for: .touchUpInside)
        rulerResetButton.addTarget(self, action: #selector(resetRulerTapped), for: .touchUpInside)
        
        view.bringSubviewToFront(webView)
        view.bringSubviewToFront(webViewButton)
        
    }


    @objc func toggleWebView() {
        if isWebViewVisible {
            // 非表示にする（フェードアウト）
            UIView.transition(with: webView, duration: 0.3, options: .transitionCrossDissolve, animations: {
                self.webView.isHidden = true
            })
        } else {
            // 表示にする（フェードイン）
            UIView.transition(with: webView, duration: 0.3, options: .transitionCrossDissolve, animations: {
                self.webView.isHidden = false
            })
        }
        isWebViewVisible.toggle()
    }
    
    @objc private func toggleBackgroundButtonTapped() {
        plotScene.toggleBackground()
    }
    
    @objc private func protractButtonTapped() {
        plotScene.toggleProtract()
        if let protractorNode = plotScene.protractor {
            let isVisible = !protractorNode.isHidden
            // ボタンのスタイルを更新（黄色の縁取りと文字色）
            protractButton.layer.borderWidth = isVisible ? 2 : 0
            protractButton.layer.borderColor = UIColor.yellow.cgColor
            protractButton.setTitleColor(isVisible ? .yellow : .white, for: .normal)
            // ラベルとリセットボタンの表示制御
            protractResetButton.isHidden = !isVisible
            angleLabel.isHidden = !isVisible
        }
    }
    
    @objc private func resetProtractorTapped() {
        guard let protractor = plotScene.protractor else { return }
        // プロトラクターの位置と回転をリセット
        protractor.position = .zero
        protractor.zRotation = 0
        // スケールをリセット
        protractor.setScale(0.5)
        // スケール変数のリセット
        currentScale = 0.5  // この初期スケールを設定
        lastScale = 0.5     // 現在のスケールを基準に初期化
        p_currentScale = 0.5
        p_lastScale = 0.5   // p_currentScaleの初期化
        angleLabel.text = "0°"  // ラベル値を0に
        
    }
    
    @objc private func rulerButtonTapped() {
        plotScene.toggleRuler()
        if let ruler = plotScene.ruler {
            let isVisible = !ruler.isHidden
            // ボタンのスタイルを更新（赤太字 or 黒通常）
            rulerButton.layer.borderWidth = isVisible ? 2 : 0
            rulerButton.layer.borderColor = UIColor.yellow.cgColor
            rulerButton.setTitleColor(isVisible ? .yellow : .white, for: .normal)
            // ラベルとリセットボタンの表示制御
            rulerResetButton.isHidden = !isVisible
            rulerAngleLabel.isHidden = !isVisible
        }
    }

    @objc private func resetRulerTapped() {
        guard let ruler = plotScene.ruler else { return }
        // 定規の位置と回転をリセット
        ruler.position = .zero
        ruler.zRotation = 0
        // ラベル更新
        rulerAngleLabel.text = "0°"
    }
    
    @objc private func handlePinch(_ sender: UIPinchGestureRecognizer) {
        switch sender.state {
        case .began:
            lastScale = currentScale
            p_lastScale = p_currentScale

        case .changed:
            let scaleDelta = sender.scale
            // 分度器がONのとき、ProtractorNodeのスケーリングを調整
            if let protractorNode = plotScene.protractor, protractorNode.isHidden == false {
                //分度器がONのとき
                let newScale = p_lastScale * scaleDelta
                let clampedScale = max(min(newScale, p_maxScale), p_minScale)
                protractorNode.setScale(clampedScale)
                p_currentScale = clampedScale
            } else {
                let newScale = lastScale * scaleDelta
                let clampedScale = max(min(newScale, maxScale), minScale)
                currentScale = clampedScale
                plotScene.handlePinch(scale: clampedScale)
            }

        default:
            break
        }
    }

    @objc private func handlePan(_ sender: UIPanGestureRecognizer) {
        guard let skView = sender.view as? SKView,
              let scene = skView.scene,
              let protractorNode = plotScene.protractor,
              let rulerNode = plotScene.ruler else { return }

        if sender.numberOfTouches == 2 {
            switch sender.state {
            case .began:
                lastTranslation = sender.translation(in: skView)

            case .changed:
                let translation = sender.translation(in: skView)
                let fromPoint = CGPoint(x: lastTranslation.x, y: lastTranslation.y)
                let toPoint = CGPoint(x: translation.x, y: translation.y)
                let fromSK = scene.convertPoint(fromView: fromPoint)
                let toSK = scene.convertPoint(fromView: toPoint)
                let delta = CGPoint(x: toSK.x - fromSK.x, y: toSK.y - fromSK.y)
                plotScene.handlePan(dx: delta.x, dy: -delta.y)
                lastTranslation = translation

            default:
                break
            }
        } else if sender.numberOfTouches == 1 {
            
            switch sender.state {
            case .began:
                lastTranslation = sender.translation(in: skView)
                
            case .changed:
                let translation = sender.translation(in: skView)
                let fromPoint = CGPoint(x: lastTranslation.x, y: lastTranslation.y)
                let toPoint = CGPoint(x: translation.x, y: translation.y)
                
                let fromSK = scene.convertPoint(fromView: fromPoint)
                let toSK = scene.convertPoint(fromView: toPoint)
                
                let delta = CGPoint(x: toSK.x - fromSK.x, y: toSK.y - fromSK.y)
                
                if !protractorNode.isHidden {
                    protractorNode.position.x += delta.x
                    protractorNode.position.y += delta.y
                }
                
                // 定規が表示されている場合、スケール補正を適用
                if !rulerNode.isHidden && protractorNode.isHidden {
                    let scale = plotScene.scaleNode.xScale
                    // deltaに対してスケール補正
                    let adjustedDelta = CGPoint(x: delta.x / scale, y: delta.y / scale)
                    rulerNode.position.x += adjustedDelta.x
                    rulerNode.position.y += adjustedDelta.y
                }
                
                if protractorNode.isHidden && rulerNode.isHidden {
                    plotScene.handlePan(dx: delta.x, dy: -delta.y)
                }
                
                lastTranslation = translation
                
            default:
                break
            }
        }
    }

    
    private var accumulatedRotation: CGFloat = 0

    @objc private func handleRotation(_ sender: UIRotationGestureRecognizer) {
        // 表示されているノードだけ回転対象にする
        let targetNode: SKNode? = {
            if let protractor = plotScene.protractor, !protractor.isHidden {
                return protractor
            } else if let ruler = plotScene.ruler, !ruler.isHidden {
                return ruler
            } else {
                return nil
            }
        }()

        guard let node = targetNode else { return }

        let rotation = sender.rotation
        accumulatedRotation += rotation

        let degrees = accumulatedRotation * 180 / .pi
        let roundedDegrees = round(degrees)

        if abs(roundedDegrees) >= 1 {
            let stepRotation = roundedDegrees * .pi / 180
            node.zRotation -= stepRotation
            accumulatedRotation -= stepRotation
        }

        // 分度器が表示中なら角度表示を更新
        if let protractor = plotScene.protractor, node === protractor {
            let totalDegrees = protractor.zRotation * 180 / .pi
            let normalizedDegrees = (round(totalDegrees).truncatingRemainder(dividingBy: 360) + 360).truncatingRemainder(dividingBy: 360)
            angleLabel.text = "\(Int(normalizedDegrees))°"
        }

        // 定規が表示中なら角度表示を更新
        if let ruler = plotScene.ruler, node === ruler {
            let totalDegrees = ruler.zRotation * 180 / .pi
            let normalizedDegrees = (round(totalDegrees).truncatingRemainder(dividingBy: 360) + 360).truncatingRemainder(dividingBy: 360)
            rulerAngleLabel.text = "\(Int(normalizedDegrees))°"
        }

        sender.rotation = 0
    }



    
    @objc private func toggleBackground() {
        plotScene.toggleBackground()
    }
    
    @objc private func togglePenVisible() {
        plotScene.togglePenVisible()
    }
    
    @objc private func showHelp() {
        let helpVC = UIViewController()
        helpVC.view.backgroundColor = .white

        let textView = UITextView()
        textView.translatesAutoresizingMaskIntoConstraints = false
        textView.isEditable = false
        textView.alwaysBounceVertical = true
        textView.backgroundColor = .white

        // ✅ RTFファイルの読み込み
        if let rtfPath = Bundle.main.path(forResource: "HelpText", ofType: "rtf"),
           let rtfData = try? Data(contentsOf: URL(fileURLWithPath: rtfPath)),
           let attributedString = try? NSAttributedString(data: rtfData,
                                                          options: [.documentType: NSAttributedString.DocumentType.rtf],
                                                          documentAttributes: nil) {
            textView.attributedText = attributedString
        } else {
            textView.text = "読み込みエラー：HelpText.rtf が見つかりませんでした。"
        }

        helpVC.view.addSubview(textView)
        NSLayoutConstraint.activate([
            textView.topAnchor.constraint(equalTo: helpVC.view.topAnchor, constant: 20),
            textView.leadingAnchor.constraint(equalTo: helpVC.view.leadingAnchor, constant: 20),
            textView.trailingAnchor.constraint(equalTo: helpVC.view.trailingAnchor, constant: -20),
            textView.bottomAnchor.constraint(equalTo: helpVC.view.bottomAnchor, constant: -20)
        ])

        let navVC = UINavigationController(rootViewController: helpVC)
        helpVC.navigationItem.title = "使い方ガイド"
        helpVC.navigationItem.rightBarButtonItem = UIBarButtonItem(barButtonSystemItem: .done, target: self, action: #selector(dismissHelp))

        self.present(navVC, animated: true, completion: nil)
    }


    @objc private func dismissHelp() {
        self.dismiss(animated: true, completion: nil)
    }
    
    func animationDidFinish() {
        //print("animationDidFinish")
        let message: PlaygroundValue = .string("done")
        send(message)
    }
}

// PlaygroundLiveViewMessageHandler プロトコルの実装をextensionで分ける
extension PlotViewController: PlaygroundLiveViewMessageHandler {
    public func liveViewMessageConnectionOpened() {
        // メッセージハンドラがオープンした時（実行ボタンが押された時）
        //ブロックビューは閉じる
        if isWebViewVisible {
            // 非表示にする（フェードアウト）
            UIView.transition(with: webView, duration: 0.3, options: .transitionCrossDissolve, animations: {
                self.webView.isHidden = true
            })
        }
        isWebViewVisible = false
    }
    
    public func receive(_ message: PlaygroundValue) {
        // メッセージが辞書形式の場合
        if case let .dictionary(dict) = message {
            // コマンドが"moveTo"である場合、x, y 座標を取得してペンの位置を更新
            if case let .string(command)? = dict["command"], command == "moveTo",
               case let .floatingPoint(x)? = dict["x"],
               case let .floatingPoint(y)? = dict["y"] {
                plotScene.moveTo(x: x, y: y)
            }
            // コマンドが"moveBy"である場合、x, y 座標を取得してペンを相対移動
            if case let .string(command)? = dict["command"], command == "moveBy",
               case let .floatingPoint(x)? = dict["x"],
               case let .floatingPoint(y)? = dict["y"] {
                // 相対座標で移動
                plotScene.moveBy(x: x, y: y)
            }
            
            // コマンドが "drawCircle" である場合、radius を取得して円を描画
            if case let .string(command)? = dict["command"], command == "drawCircle",
               case let .floatingPoint(radius)? = dict["radius"] {
                plotScene.drawCircle(radius: radius)
            }
            
            if case let .string(command)? = dict["command"], command == "move",
                    case let .floatingPoint(distance)? = dict["distance"] {
                plotScene.move(distance: distance)
            }
            
            if case let .string(command)? = dict["command"], command == "angle",
               case let .floatingPoint(value)? = dict["value"] {
                plotScene.setAngle(value)
            }
            
            if case let .string(command)? = dict["command"], command == "turn",
               case let .floatingPoint(angle)? = dict["angle"] {
                plotScene.penTurn(by: angle)
            }

            //コマンドが"speed"の場合
            if case let .string(command)? = dict["command"], command == "speed",
               case let .floatingPoint(value)? = dict["value"] {
                plotScene.setSpeed(to: value)
            }
            
            //コマンドが"on"の場合
            if case let .string(command)? = dict["command"], command == "on" {
                plotScene.penDown()
            }
            //コマンドが"off"の場合
            if case let .string(command)? = dict["command"], command == "off" {
                plotScene.penUp()
            }
            
            //コマンドが"clear"の場合
            if case let .string(command)? = dict["command"], command == "clear" {
                plotScene.clearPlot()
            }
            
            //コマンドが"color"の場合
            if case let .string(command)? = dict["command"], command == "color",
               case let .string(colorName)? = dict["value"] {
                plotScene.setPenColor(named: colorName)
            }
            
            //コマンドが"width"の場合
            if case let .string(command)? = dict["command"], command == "width",
               case let .floatingPoint(value)? = dict["value"] {
                plotScene.setPenWidth(to: value)
            }
        }
    }
}

// PlotSceneクラス
class PlotScene: SKScene {
    weak var plotDelegate: PlotSceneDelegate?
    private var pen: SKSpriteNode!
    public var scaleNode = SKNode()
    private var isDarkMode = false
    private var penIsDown = false
    private var penSpeed = 100.0
    private var penAngle = 0.0
    private var penColor: SKColor = .red
    private var penWidth: CGFloat = 3.0
    public var protractor: SKSpriteNode!
    public var ruler: SKSpriteNode!
    private var draggingNode: SKNode?
    
    override func didMove(to view: SKView) {
        self.anchorPoint = CGPoint(x: 0.5, y: 0.5)
        addChild(scaleNode)
        //drawAxes()
        drawGrid()
        pen = SKSpriteNode(imageNamed: "pen_off")
        pen.anchorPoint = CGPoint(x: 0.5, y: 0.0)  // 横中央・下端を原点に
        pen.position = .zero
        pen.setScale(0.5)  // サイズを半分に
        backgroundColor = .white  // ← 初期背景を白に設定
        scaleNode.addChild(pen)
        protractor = ProtractorNode()
        protractor.position = CGPoint(x: 0, y: 0)
        protractor.zPosition = 1001
        protractor.isUserInteractionEnabled = true
        protractor.isHidden = true
        addChild(protractor)
        ruler = RulerNode()
        ruler.position = CGPoint(x: 0, y: 0)
        ruler.zPosition = 1000
        ruler.isUserInteractionEnabled = true
        ruler.isHidden = true
        scaleNode.addChild(ruler)
    }

    func togglePenVisible() {
        pen.isHidden.toggle()
    }
    
    func clearPlot() {
        scaleNode.removeAllChildren()
        drawGrid()
        scaleNode.addChild(pen)
        scaleNode.addChild(ruler)
        plotDelegate?.animationDidFinish()
    }

    func drawShape(at position: CGPoint) {
        let shape = SKShapeNode(circleOfRadius: 10)
        shape.fillColor = isDarkMode ? .white : .black
        shape.position = position
        scaleNode.addChild(shape)
    }

    func handlePinch(scale: CGFloat) {
        scaleNode.setScale(scale)
    }
    
    func handlePan(dx: CGFloat, dy: CGFloat) {
        let scale = scaleNode.xScale  // 現在のスケールを取得

        // スケールに応じて移動範囲を拡大（等倍時 ±800 → スケール倍の範囲）
        let scaledMinX: CGFloat = -800 * scale
        let scaledMaxX: CGFloat = 800 * scale
        let scaledMinY: CGFloat = -800 * scale
        let scaledMaxY: CGFloat = 800 * scale

        // 現在位置を基準に移動（Yは符号反転）
        var newX = scaleNode.position.x + dx
        var newY = scaleNode.position.y - dy

        // 移動範囲を制限
        newX = min(max(newX, scaledMinX), scaledMaxX)
        newY = min(max(newY, scaledMinY), scaledMaxY)

        scaleNode.position = CGPoint(x: newX, y: newY)
    }
    
    // 移動メソッドを追加
    func moveScaleNode(dx: CGFloat, dy: CGFloat) {
        scaleNode.position = CGPoint(x: scaleNode.position.x + dx, y: scaleNode.position.y + dy)
    }

    func toggleBackground() {
        isDarkMode.toggle()
        self.backgroundColor = isDarkMode ? .black : .white
    }
    
    func toggleProtract() {
        protractor.isHidden.toggle()
    }
    
    func toggleRuler() {
        ruler.isHidden.toggle()
    }
    
    func setSpeed(to value: Double) {
        // スピードの範囲を 1〜9999 に制限
        let clampedValue = max(1.0, min(9999.0, value))
        penSpeed = clampedValue
        plotDelegate?.animationDidFinish()
    }
    
    func setPenWidth(to value: Double) {
        // 線幅の範囲を 0.01〜999 に制限
        let clampedValue = max(0.01, min(999.0, value))
        penWidth = clampedValue
        plotDelegate?.animationDidFinish()
    }
    
    func setPenColor(named name: String) {
        switch name.lowercased() {
        case "black":  penColor = .black
        case "red":    penColor = .red
        case "green":  penColor = .green
        case "blue":   penColor = .blue
        case "cyan":   penColor = .cyan
        case "yellow": penColor = .yellow
        case "orange": penColor = .orange
        case "purple": penColor = .purple
        case "brown":  penColor = .brown
        default: break
        }
        plotDelegate?.animationDidFinish()
    }
    
    func penDown() {
        let textureName: String
        switch penColor {
        case UIColor.red:    textureName = "pen_on_red"
        case UIColor.green:  textureName = "pen_on_green"
        case UIColor.blue:   textureName = "pen_on_blue"
        case UIColor.cyan:   textureName = "pen_on_cyan"
        case UIColor.yellow: textureName = "pen_on_yellow"
        case UIColor.orange: textureName = "pen_on_orange"
        case UIColor.purple: textureName = "pen_on_purple"
        case UIColor.brown:  textureName = "pen_on_brown"
        case UIColor.black:  textureName = "pen_on_black"
        default:             textureName = "pen_on_black" // フォールバック
        }
        pen.texture = SKTexture(imageNamed: textureName)
        penIsDown = true
        plotDelegate?.animationDidFinish()
    }
    
    func penUp() {
        pen.texture = SKTexture(imageNamed: "pen_off")
        penIsDown = false
        plotDelegate?.animationDidFinish()
    }
    
    // ペンの角度を指定する
    func setAngle(_ angle: CGFloat) {
        // 最短回転方向を計算
        var angleDifference = angle - penAngle
        if angleDifference > 180 {
            angleDifference -= 360
        } else if angleDifference < -180 {
            angleDifference += 360
        }

        penAngle = angle

        // アニメーションで回転させる
        let radians = angleDifference * .pi / 180
        let rotateAction = SKAction.rotate(byAngle: radians, duration: 0.3) // 任意の速さで回転

        pen.run(SKAction.sequence([
            rotateAction,
            .run { [weak self] in self?.plotDelegate?.animationDidFinish() }
        ]))
    }
    
    // 移動量＋角度でペンの位置を一定の速度で移動させる
    func move(distance: CGFloat) {
        // 負の距離は無効として処理
        guard distance > 0 else {
            print("警告: move(distance:) に負の値が渡されました（無視されます）")
            plotDelegate?.animationDidFinish()
            return
        }
        
        let startPoint = pen.position
        // penAngle（度）をラジアンに変換して移動量を計算
        let radians = penAngle * .pi / 180
        let dx = cos(radians) * distance
        let dy = sin(radians) * distance
        let endPoint = CGPoint(x: startPoint.x + dx, y: startPoint.y + dy)

        let duration = distance / penSpeed
        let moveAction = SKAction.move(to: endPoint, duration: duration)

        if penIsDown {
            let path = CGMutablePath()
            path.move(to: startPoint)

            let shapeNode = SKShapeNode()
            shapeNode.path = path
            shapeNode.strokeColor = penColor
            shapeNode.lineWidth = penWidth
            shapeNode.lineCap = .round
            shapeNode.zPosition = -1
            scaleNode.addChild(shapeNode)

            var lastPoint = startPoint
            let updateAction = SKAction.customAction(withDuration: duration) { [weak self] node, elapsedTime in
                guard self != nil else { return }
                let t = min(max(CGFloat(elapsedTime / CGFloat(duration)), 0), 1)
                let currentX = startPoint.x + dx * t
                let currentY = startPoint.y + dy * t
                let currentPoint = CGPoint(x: currentX, y: currentY)

                path.move(to: lastPoint)
                path.addLine(to: currentPoint)
                shapeNode.path = path
                lastPoint = currentPoint
            }

            let group = SKAction.group([moveAction, updateAction])

            pen.run(SKAction.sequence([
                group,
                .run { [weak self] in self?.plotDelegate?.animationDidFinish() }
            ]))
        } else {
            pen.run(SKAction.sequence([
                moveAction,
                .run { [weak self] in self?.plotDelegate?.animationDidFinish() }
            ]))
        }
    }
    
    // PlotScene ペンの角度を回転させる
    func penTurn(by angle: CGFloat) {
        penAngle += angle
        let radians = angle * .pi / 180
        let duration = abs(angle) / penSpeed

        let rotateAction = SKAction.rotate(byAngle: radians, duration: duration)

        pen.run(SKAction.sequence([
            rotateAction,
            .run { [weak self] in
                guard let self = self else { return }
                self.penAngle = self.penAngle.truncatingRemainder(dividingBy: 360)
                if self.penAngle < 0 { self.penAngle += 360 }
                self.plotDelegate?.animationDidFinish()
            }
        ]))
    }


    
    // PlotScene 絶対座標でペンの位置を一定の速度で移動させる
    func moveTo(x: CGFloat, y: CGFloat) {
        let startPoint = pen.position
        let endPoint = CGPoint(x: x, y: y)
        let dx = endPoint.x - startPoint.x
        let dy = endPoint.y - startPoint.y
        let distance = hypot(dx, dy)
        let duration = distance / penSpeed

        let moveAction = SKAction.move(to: endPoint, duration: duration)

        // 軌跡を描く処理
        if penIsDown {
            let path = CGMutablePath()
            path.move(to: startPoint)

            let shapeNode = SKShapeNode()
            shapeNode.path = path
            shapeNode.strokeColor = penColor
            shapeNode.lineWidth = penWidth
            shapeNode.lineCap = .round
            shapeNode.zPosition = -1
            scaleNode.addChild(shapeNode)

            var lastPoint = startPoint
            let updateAction = SKAction.customAction(withDuration: duration) { [weak self] node, elapsedTime in
                guard self != nil else { return }
                let t = min(max(CGFloat(elapsedTime / CGFloat(duration)), 0), 1)
                let newX = startPoint.x + dx * t
                let newY = startPoint.y + dy * t
                let currentPoint = CGPoint(x: newX, y: newY)

                path.move(to: lastPoint)
                path.addLine(to: currentPoint)
                shapeNode.path = path
                lastPoint = currentPoint
            }

            let group = SKAction.group([moveAction, updateAction])

            pen.run(SKAction.sequence([
                group,
                .run { [weak self] in self?.plotDelegate?.animationDidFinish() }
            ]))
        } else {
            // 軌跡なしで移動のみ
            pen.run(SKAction.sequence([
                moveAction,
                .run { [weak self] in self?.plotDelegate?.animationDidFinish() }
            ]))
        }
    }
    
    // PlotScene 相対座標でペンを一定の速度で移動させる
    func moveBy(x: CGFloat, y: CGFloat) {
        let startPoint = pen.position
        let endPoint = CGPoint(x: startPoint.x + x, y: startPoint.y + y)
        let dx = endPoint.x - startPoint.x
        let dy = endPoint.y - startPoint.y
        let distance = hypot(dx, dy)
        let duration = distance / penSpeed

        let moveAction = SKAction.move(to: endPoint, duration: duration)

        if penIsDown {
            let path = CGMutablePath()
            path.move(to: startPoint)

            let shapeNode = SKShapeNode()
            shapeNode.path = path
            shapeNode.strokeColor = penColor
            shapeNode.lineWidth = penWidth
            shapeNode.lineCap = .round
            shapeNode.zPosition = -1
            scaleNode.addChild(shapeNode)

            var lastPoint = startPoint
            let updateAction = SKAction.customAction(withDuration: duration) { [weak self] node, elapsedTime in
                guard self != nil else { return }
                let t = min(max(CGFloat(elapsedTime / CGFloat(duration)), 0), 1)
                let newX = startPoint.x + dx * t
                let newY = startPoint.y + dy * t
                let currentPoint = CGPoint(x: newX, y: newY)

                path.move(to: lastPoint)
                path.addLine(to: currentPoint)
                shapeNode.path = path
                lastPoint = currentPoint
            }

            let group = SKAction.group([moveAction, updateAction])

            pen.run(SKAction.sequence([
                group,
                .run { [weak self] in self?.plotDelegate?.animationDidFinish() }
            ]))
        } else {
            // 軌跡なしで移動のみ
            pen.run(SKAction.sequence([
                moveAction,
                .run { [weak self] in self?.plotDelegate?.animationDidFinish() }
            ]))
        }
    }
    
    func drawCircle(radius: CGFloat, steps: Int = 120) {
        guard radius > 0 else {
            print("警告: drawCircle に負の半径が渡されました（無視されます）")
            plotDelegate?.animationDidFinish()
            return
        }

        // centerはpen.positionに半径を加えた位置として設定
        let center = CGPoint(x: pen.position.x - radius, y: pen.position.y)
        let startAngle = penAngle * .pi / 180
        let totalAngle = 2 * CGFloat.pi
        let anglePerStep = totalAngle / CGFloat(steps)

        let circumference = 2 * .pi * radius
        let totalDuration = circumference / penSpeed
        let durationPerStep = totalDuration / CGFloat(steps)

        let path = CGMutablePath()
        let shapeNode = SKShapeNode()
        shapeNode.strokeColor = penColor
        shapeNode.lineWidth = penWidth
        shapeNode.lineCap = .round
        shapeNode.zPosition = -1
        scaleNode.addChild(shapeNode)

        var actions: [SKAction] = []
        var lastPoint: CGPoint? = nil

        // 現在位置を円周上の点に設定
        let firstPoint = CGPoint(
            x: center.x + radius * cos(startAngle),
            y: center.y + radius * sin(startAngle)
        )
        pen.position = firstPoint  // ペンを円周上の点に移動
        lastPoint = firstPoint
        path.move(to: firstPoint)

        // 各ステップごとのアクションを作成
        for step in 1..<steps {
            let angle = startAngle + CGFloat(step) * anglePerStep
            let nextPoint = CGPoint(
                x: center.x + radius * cos(angle),
                y: center.y + radius * sin(angle)
            )

            let drawStep = SKAction.run { [weak self] in
                guard let self = self else { return }

                if self.penIsDown, let last = lastPoint {
                    path.move(to: last)
                    path.addLine(to: nextPoint)
                    shapeNode.path = path
                }

                self.pen.position = nextPoint
                lastPoint = nextPoint
            }

            actions.append(drawStep)
            actions.append(SKAction.wait(forDuration: durationPerStep))
        }

        // 最後に最初の点に戻る処理
        let closeCircle = SKAction.run { [weak self] in
            guard let self = self else { return }

            if self.penIsDown, let last = lastPoint {
                path.move(to: last)
                path.addLine(to: firstPoint)  // 最初の点に戻る
                shapeNode.path = path
            }

            self.pen.position = firstPoint
            lastPoint = firstPoint
        }

        actions.append(closeCircle)

        // 最後にアニメーション終了のコールバックを追加
        actions.append(SKAction.run { [weak self] in
            self?.plotDelegate?.animationDidFinish()
        })

        pen.run(SKAction.sequence(actions))
    }
    
    private func drawPath(from start: CGPoint, to end: CGPoint) {
        let path = CGMutablePath()
        path.move(to: start)
        path.addLine(to: end)
        
        let line = SKShapeNode(path: path)
        line.strokeColor = penColor
        line.lineWidth = penWidth
        line.zPosition = -1  // ペンより奥に表示
        scaleNode.addChild(line)
    }
    
    private func drawAxes() {
        // X軸
        let pathX = CGMutablePath()
        pathX.move(to: CGPoint(x: -frame.width / 2, y: 0)) // 中心から描画
        pathX.addLine(to: CGPoint(x: frame.width / 2, y: 0))
        
        let xAxis = SKShapeNode(path: pathX)
        xAxis.strokeColor = .gray
        xAxis.lineWidth = 1.0
        scaleNode.addChild(xAxis)

        // Y軸
        let pathY = CGMutablePath()
        pathY.move(to: CGPoint(x: 0, y: -frame.height / 2)) // 中心から描画
        pathY.addLine(to: CGPoint(x: 0, y: frame.height / 2))
        
        let yAxis = SKShapeNode(path: pathY)
        yAxis.strokeColor = .gray
        yAxis.lineWidth = 1.0
        scaleNode.addChild(yAxis)

        // X軸目盛り（50間隔）
        for i in stride(from: -Int(frame.width / 2), through: Int(frame.width / 2), by: 50) {
            if i == 0 { continue }
            let tick = SKShapeNode(rectOf: CGSize(width: 1, height: 5))
            tick.strokeColor = .gray
            tick.position = CGPoint(x: CGFloat(i), y: 0)
            tick.xScale = 1 / scaleNode.xScale  // スケーリングを逆に適用
            scaleNode.addChild(tick)
        }

        // Y軸目盛り（50間隔）
        for i in stride(from: -Int(frame.height / 2), through: Int(frame.height / 2), by: 50) {
            if i == 0 { continue }
            let tick = SKShapeNode(rectOf: CGSize(width: 5, height: 1))
            tick.strokeColor = .gray
            tick.position = CGPoint(x: 0, y: CGFloat(i))
            tick.zPosition = -10 // 格子を背景に設定
            tick.yScale = 1 / scaleNode.yScale  // スケーリングを逆に適用
            scaleNode.addChild(tick)
        }
    }
    
    private func drawGrid() {
        // ペンの移動範囲に基づく格子の描画
        let gridSize: CGFloat = 50  // 目盛りの間隔を固定
        let expansionFactor: CGFloat = 50.0  // 格子を外側に5倍拡大

        // X軸目盛り
        for i in stride(from: -Int(gridSize * expansionFactor), through: Int(gridSize * expansionFactor), by: Int(gridSize)) {
            let tick = SKShapeNode(rectOf: CGSize(width: 1, height: 5))
            tick.strokeColor = .gray
            tick.position = CGPoint(x: CGFloat(i), y: 0)
            tick.zPosition = -10 // 格子線を背景奥に
            scaleNode.addChild(tick)
        }

        // Y軸目盛り
        for i in stride(from: -Int(gridSize * expansionFactor), through: Int(gridSize * expansionFactor), by: Int(gridSize)) {
            let tick = SKShapeNode(rectOf: CGSize(width: 5, height: 1))
            tick.strokeColor = .gray
            tick.position = CGPoint(x: 0, y: CGFloat(i))
            tick.zPosition = -10 // 格子線を背景奥に
            scaleNode.addChild(tick)
        }

        // 縦中央線
        let pathX = CGMutablePath()
        pathX.move(to: CGPoint(x: 0, y: -gridSize * expansionFactor))
        pathX.addLine(to: CGPoint(x: 0, y: gridSize * expansionFactor))

        let verticalLine = SKShapeNode(path: pathX)
        verticalLine.strokeColor = .gray
        verticalLine.lineWidth = 4.0  // 太い線
        verticalLine.zPosition = -10 // 格子線を背景奥に
        scaleNode.addChild(verticalLine)

        // 横中央線
        let pathY = CGMutablePath()
        pathY.move(to: CGPoint(x: -gridSize * expansionFactor, y: 0))
        pathY.addLine(to: CGPoint(x: gridSize * expansionFactor, y: 0))

        let horizontalLine = SKShapeNode(path: pathY)
        horizontalLine.strokeColor = .gray
        horizontalLine.lineWidth = 4.0  // 太い線
        horizontalLine.zPosition = -10 // 格子線を背景奥に
        scaleNode.addChild(horizontalLine)

        // 格子線（縦線）
        for i in stride(from: -Int(gridSize * expansionFactor), through: Int(gridSize * expansionFactor), by: Int(gridSize)) {
            let path = CGMutablePath()
            path.move(to: CGPoint(x: CGFloat(i), y: -gridSize * expansionFactor))
            path.addLine(to: CGPoint(x: CGFloat(i), y: gridSize * expansionFactor))
            
            let line = SKShapeNode(path: path)
            line.strokeColor = .lightGray  // 中央以外は薄い色
            line.lineWidth = 0.5  // 細い線
            line.zPosition = -10 // 格子線を背景奥に
            scaleNode.addChild(line)
        }

        // 格子線（横線）
        for i in stride(from: -Int(gridSize * expansionFactor), through: Int(gridSize * expansionFactor), by: Int(gridSize)) {
            let path = CGMutablePath()
            path.move(to: CGPoint(x: -gridSize * expansionFactor, y: CGFloat(i)))
            path.addLine(to: CGPoint(x: gridSize * expansionFactor, y: CGFloat(i)))
            
            let line = SKShapeNode(path: path)
            line.strokeColor = .lightGray  // 中央以外は薄い色
            line.lineWidth = 0.5  // 細い線
            line.zPosition = -10 // 格子線を背景奥に
            scaleNode.addChild(line)
        }
    }
}


class ProtractorNode: SKSpriteNode {
    convenience init() {
        let texture = SKTexture(imageNamed: "protractor")
        self.init(texture: texture, color: .clear, size: texture.size())
        self.setScale(0.5)
        self.zPosition = 1001
    }

    override init(texture: SKTexture?, color: UIColor, size: CGSize) {
        super.init(texture: texture, color: color, size: size)
    }

    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
}

class RulerNode: SKSpriteNode {
    convenience init() {
        let texture = SKTexture(imageNamed: "ruler")
        self.init(texture: texture, color: .clear, size: texture.size())
        self.setScale(0.5)
        self.zPosition = 1000
    }

    override init(texture: SKTexture?, color: UIColor, size: CGSize) {
        super.init(texture: texture, color: color, size: size)
    }

    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
}
